{******************************************************}
{            Slim Dynamic Array Class v3.0             }
{        Copyright 1997 RealSoft Development           }
{           support:  www.realsoftdev.com              }
{                    ------------                      }
{ This is a slim version on the Dynarray Class         }
{ to be distributed with your source code.  RealSoft   }
{ grants an unrestricted license to include this unit  }
{ in its un modified format.  For a full featured      }
{ version, contact dan@realSoftdev.com, or visit the   }
{ Compuserve Delphi forum.  Do not remove this notice. }
{******************************************************}

unit Dynslim;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Dialogs;

type
  EDynArrayException    = class(Exception);
  EDynArrayCreateError  = class(EDynArrayException);
  EDynArrayIndexBounds  = class(EDynArrayException);
  EDynArrayResources    = class(EDynArrayException);
  EDynArrayFileError    = class(EDynArrayException);
  EDynArrayAddError     = class(EDynArrayException);
  EDynArrayDelError     = class(EDynArrayException);

  TDynArray = class
    private
      FPointer     : Pointer;
      FItemSize    : Longint;
      FItemCount   : Longint;
      FArraySize   : LongInt;
      procedure SetItems(Index : Longint; AValue : pointer);
      function  GetItems (Index : Longint) : pointer;
    public
      constructor Create ( ItemSize : Longint );
      destructor  Destroy; override;
      function    Clear  : Pointer;
      function    Add    ( const Item ) : Pointer;
      function    Delete ( Index : Longint  ) : Pointer;
      function    SaveToFile   ( filename : String ) : Pointer;
      function    LoadFromFile ( filename : String ) : Pointer;
      property Count       : Longint   read FItemCount;
      property Size        : Longint   read FArraySize;
      property ItemSize    : Longint   read FItemSize;
      property DataPtr     : Pointer   read FPointer;
      property Items[Index: longint]: Pointer read GetItems write SetItems; default;
    end;

implementation

{***********************}
{  Create & Initialize  }
{***********************}
constructor TDynArray.Create( ItemSize : Longint );
begin
  inherited create;
  if (ItemSize > 0) and (ItemSize < 65520) then begin
    FItemCount:= 0;
    FArraySize:= 0;
    FItemSize:= ItemSize;
    FPointer:= nil;
    end
  else raise EDynArrayCreateError.Create('Dynamic Array: Invalid Item Size');
end;

{***********************}
{    Destroy & Free     }
{***********************}
destructor TDynArray.Destroy;
begin
  Clear;
  FItemSize:= 0;
  inherited destroy;
end;

{***********************}
{      Clear Array      }
{***********************}
function TDynArray.Clear : pointer;
begin
  if FItemCount > 0 then begin
    FreeMem(FPointer, FArraySize);
    FItemCount:= 0;
    FArraySize:= 0;
    end;
  result:= NIL;
end;

{***********************}
{ Add an Array Element  }
{***********************}
function TDynArray.Add ( const Item ) : Pointer;
var P : Pointer;
begin
  if FItemSize > 0 then begin
    {Allocate next memory element}
    if FItemCount = 0 then GetMem( FPointer, FItemSize )
    else
    {$IFDEF Win32}
      ReAllocMem( FPointer, FArraySize + FItemSize );
    {$ELSE}
      FPointer:= ReAllocMem( FPointer, FArraySize, FArraySize + FItemSize );
    {$ENDIF}

    if FPointer <> nil then begin {check for valid pointer}
      {advance counters}
      inc(FItemCount);
      inc(FArraySize, FItemSize);
      {move data into array memory}
      P:= FPointer;
      inc( longint(P), (FItemSize * (FItemCount - 1)) );
      move( Item, P^, FItemSize );
      end
    else raise EDynArrayResources.Create('Dynamic Array: Out of resources during Add.');
    end
  else begin
    raise EDynArrayAddError.Create('Dynamic Array: Unable to add element.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{**********************}
{ Del an Array Element }
{**********************}
function TDynArray.Delete( Index: Longint ) : Pointer;
var
  x        : smallint;
  P1 : Pointer;
  P2 : Pointer;
begin
  if FItemCount > 0 then begin
    if (Index < FItemCount - 1) then begin
      {move items to fill gap}
      P1:= FPointer;
      inc( longint(P1), FItemSize * Index );
      P2:= FPointer;
      inc( longint(P2), FItemSize * (Index + 1) );
      for x:= Index to FItemCount - 2 do begin
        move( P2^, P1^, FItemSize );
        inc( longint(P1), FItemSize );
        inc( longint(P2), FItemSize );
        end;
      end;
    {resize array to clip last item}

    {$IFDEF Win32}
      ReAllocMem( FPointer, FArraySize - FItemSize );
    {$ELSE}
      Fpointer:= ReAllocMem( FPointer, FArraySize, FArraySize - FItemSize );
    {$ENDIF}

    Dec(FArraySize, FItemSize);
    Dec(FItemCount);
    end
  else begin
    raise EDynArrayDelError.Create('Dynamic Array: Unable to delete element.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{*********************}
{ Save Array to File  }
{*********************}
function TDynArray.SaveToFile ( filename : String ) : Pointer;
var
  handle, x   : smallint;
  P     : Pointer;
begin
  if FItemCount > 0 then begin
    if fileexists( FileName ) then

      {$IFDEF Win32}
        DeleteFile( PChar(FileName) );
      {$ELSE}
        DeleteFile( FileName );
      {$ENDIF}

    handle:= FileCreate( FileName );
    if handle > - 1 then begin
      P := FPointer;
      for x:= 0 to FItemCount - 1 do begin
        FileWrite( handle, P^, FItemSize );
        inc( longint(P), FItemSize );
        end;
      FileClose(handle);
      end
    else raise EDynArrayFileError.Create('Dynamic Array: Unable to create file.');
    end
  else begin
    raise EDynArrayFileError.Create('Dynamic Array: No elements to save.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{*********************}
{Load Array from File }
{*********************}
function TDynArray.LoadFromFile ( filename : String ) : Pointer;
var
  handle, x   : smallint;
  tmpptr      : Pointer;
begin
  if FItemSize > 0 then begin
    if fileexists( FileName ) then begin
      if FItemCount > 0 then begin
        FreeMem(FPointer, FArraySize);
        FItemCount:= 0;
        FArraySize:= 0;
        FPointer:= nil;
        end;
      handle:= FileOpen( FileName, 0 );

      GetMem( tmpptr, FItemSize );

      while ( FileRead( handle, tmpptr^, FItemSize ) = FItemSize ) do
        FPointer:= Add(tmpptr^);
      FileClose(handle);
      {clean up}
      FreeMem( tmpptr, FItemSize );
      end
    else raise EDynArrayFileError.Create('Dynamic Array: File does not exist.');
  end
  else begin
    raise EDynArrayFileError.Create('Dynamic Array: Element size unknown.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{*********************}
{  Item Array Access  }
{*********************}
procedure TDynArray.SetItems(Index : Longint; AValue : Pointer);
var P : pointer;
begin
  if Index > FItemCount-1 then begin
    raise EDynArrayIndexBounds.Create('Dynamic Array: Index out of bounds.');
    Exit;
    end;
  P:= FPointer;
  inc(longint(P), longint(Index * FItemSize));
  move( AValue^, P^, FItemSize );
end;

function TDynArray.GetItems(Index : Longint) : pointer;
var P : pointer;
begin
  if Index > FItemCount-1 then begin
    raise EDynArrayIndexBounds.Create('Dynamic Array: Index out of bounds.');
    Exit;
    end;
  P:= FPointer;
  inc(longint(P), longint(Index * FItemSize));
  Result:= pointer(P^);
end;

end.
